{ i2c - Simple library to access the I2C bus with a 7405 hex inverter

  Copyright (C) 2003- John Coppens

  Check http://www.jcoppens.com for updates, and more documentation

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA
}
unit i2c;

interface

const
  READ_FLG = $01;
  debug: boolean = false;

type
  i2c_port = object
    lpt: word;

    procedure Initialize(P: word);
    procedure Start_Cond;
    procedure Stop_Cond;
    function  Byte_Sent(B: byte): boolean;
    function  Receive_Byte(MustAck: boolean): byte;
    function  Poll_Available(addr: word): boolean;
    procedure Set_Debug(D: boolean);

  private
    function  Clock: boolean;
    procedure SetSDA;
    procedure ClearSDA;
    function  GetSDA: boolean;
    procedure SetSCL;
    procedure ClearSCL;
    function  GetSCL: boolean;
  end;

implementation

uses
  crt;

const
  TF: array [false..true] of char = ('0', '1');

function RdTimer: word;
var W: word;
begin
  port[$43] := $00;
  W := port[$40];
  W := W + port[$40]*256;
  RdTimer := W;
end;


procedure Wait(T: longint);
var
  Act, Ant: word;
begin
  Ant := RdTimer;
  repeat
    Act := RdTimer;
    dec(T, word (Ant-Act));
    Ant := Act;
  until T <= 0;
end;


procedure i2c_port.SetSDA;
begin
  port[lpt] := 0;
end;

procedure i2c_port.ClearSDA;
begin
  port[lpt] := $80;
end;

function i2c_port.GetSDA: boolean;
begin
  GetSDA := (port[lpt+1] and $80) <> 0;
end;

procedure i2c_port.SetSCL;
begin
  port[lpt+2] := $08;
end;

procedure i2c_port.ClearSCL;
begin
  port[lpt+2] := 0;
end;

function i2c_port.GetSCL: boolean;
begin
  GetSCL := (port[lpt+1] and $08) <> 0;
end;

function i2c_port.Clock: boolean;
var
  SDA: boolean;
begin
  SetSCL;
  Wait(1);
  repeat until (GetSCL);
  Wait(3);
  SDA := GetSDA;
  Wait(1);
  ClearSCL;
  Wait(4);
  if Debug then write('[Clck ', TF[SDA], ']');
  Clock := SDA;
end;



procedure i2c_port.Initialize(P: word);
begin
  lpt := P;
  SetSDA;
  SetSCL;
  if Debug then begin
    clrscr;
    writeln('[Init]');
  end;
end;

procedure i2c_port.Start_Cond;
begin
  SetSCL;
  Wait(5);
  ClearSDA;
  Wait(5);
  ClearSCL;
  Wait(4);
  if Debug then writeln('[Strt]');
end;

procedure i2c_port.Stop_Cond;
begin
  Wait(4);
  ClearSDA;
  SetSCL;
  Wait(5);
  SetSDA;
  Wait(10);
  if Debug then writeln('[Stop]');
end;

function i2c_port.Byte_Sent(B: byte): boolean;
var
  Bit: integer;
  tB: byte;
  Res: boolean;
begin
  tB := B;
  for Bit := 7 downto 0 do begin
    Wait(1);
    if (tB and $80) <> 0 then SetSDA else ClearSDA;
    if Clock then ;			{ Ignore the result }
    tB := tB shl 1;
  end;
  SetSDA;
  Wait(2);
  Res := not Clock;
  Byte_Sent := Res;
  if Debug then write('[Send ', B, ' ', Res, ']');
end;

function i2c_port.Receive_Byte(MustAck: boolean): byte;
var
  Bit: integer;
  Val: byte;
begin
  SetSDA;
  Wait(2);
  for Bit := 7 downto 0 do begin
    Val := Val shl 1;
    if Clock then inc(Val);
  end;
  if MustAck then begin
    ClearSDA;
    if Clock then ;
  end;
  SetSDA;
  Receive_Byte := Val;
  if Debug then write('[Recv ', Val, ' ', MustAck, ']');
end;

function i2c_port.Poll_Available(addr: word): boolean;
begin
  start_cond;
  Poll_Available := Byte_Sent(addr);
  stop_cond;
  delay(3);
end;

procedure i2c_port.Set_Debug(D: boolean);
begin
  Debug := D;
end;

begin
end.
